// ========================================
// GESTION DES GRAPHIQUES - VERSION TV RÉTRO
// ========================================

let mainChart = null;
let currentChartIndex = 0;
let currentYearIndex = 0;
let years = [];

// Palette de couleurs TV Rétro
const TV_COLORS = {
    female: '#ff6b8b',
    male: '#4a9eff',
    radio: '#50c878',
    text: '#1a1a1a',
    grid: 'rgba(0, 0, 0, 0.1)',
    accent: '#0066aa',
    background: 'rgba(255, 255, 255, 0.3)'
};

// Récupérer les années disponibles
function initYears() {
    const yearSelect = document.getElementById('yearFilter');
    if (yearSelect && yearSelect.options.length > 0) {
        years = Array.from(yearSelect.options).map(opt => opt.value);
        currentYearIndex = years.indexOf(yearSelect.value);
        console.log('Années disponibles:', years);
    }
}

// Configuration des graphiques
const chartConfigs = [
    {
        type: 'bar',
        title: 'Temps de parole par chaîne TV',
        mediaType: 'tv',
        getData: function(year) {
            const tvData = window.CSV_DATA?.years?.tv?.[year];
            if (!tvData) return generateMockTVData();
            
            const channels = Object.keys(tvData).slice(0, 8);
            if (channels.length === 0) return generateMockTVData();
            
            return {
                labels: channels,
                datasets: [
                    {
                        label: 'Femmes',
                        data: channels.map(ch => tvData[ch] || 0),
                        backgroundColor: TV_COLORS.female,
                        borderRadius: 4
                    },
                    {
                        label: 'Hommes',
                        data: channels.map(ch => 100 - (tvData[ch] || 0)),
                        backgroundColor: TV_COLORS.male,
                        borderRadius: 4
                    }
                ]
            };
        }
    },
    {
        type: 'line',
        title: 'Évolution temporelle TV',
        mediaType: 'tv',
        getData: function(year) {
            const baseYear = parseInt(year) || 2019;
            return {
                labels: [baseYear-4, baseYear-3, baseYear-2, baseYear-1, baseYear, baseYear+1].slice(0, 6),
                datasets: [
                    {
                        label: 'Femmes',
                        data: [42, 43, 45, 44, 46, 48],
                        borderColor: TV_COLORS.female,
                        backgroundColor: 'rgba(255, 107, 139, 0.15)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.3,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    },
                    {
                        label: 'Hommes',
                        data: [58, 57, 55, 56, 54, 52],
                        borderColor: TV_COLORS.male,
                        backgroundColor: 'rgba(74, 158, 255, 0.15)',
                        borderWidth: 2,
                        fill: true,
                        tension: 0.3,
                        pointRadius: 4,
                        pointHoverRadius: 6
                    }
                ]
            };
        }
    },
    {
        type: 'radar',
        title: 'Comparaison chaînes TV',
        mediaType: 'tv',
        getData: function(year) {
            const channels = ['TF1', 'France 2', 'France 3', 'Canal+', 'ARTE', 'M6'];
            return {
                labels: channels,
                datasets: [{
                    label: ' Femmes (%)',
                    data: [45, 48, 42, 50, 52, 46],
                    backgroundColor: 'rgba(255, 107, 139, 0.25)',
                    borderColor: TV_COLORS.female,
                    borderWidth: 2,
                    pointBackgroundColor: TV_COLORS.female,
                    pointRadius: 5
                }]
            };
        }
    },
    {
        type: 'bar',
        title: 'Temps de parole par chaîne Radio',
        mediaType: 'radio',
        getData: function(year) {
            const channels = ['France Inter', 'RTL', 'Europe 1', 'France Info', 'RMC', 'Skyrock'];
            return {
                labels: channels,
                datasets: [
                    {
                        label: 'Femmes',
                        data: [44, 40, 42, 45, 38, 43],
                        backgroundColor: TV_COLORS.radio,
                        borderRadius: 4
                    },
                    {
                        label: 'Hommes',
                        data: [56, 60, 58, 55, 62, 57],
                        backgroundColor: TV_COLORS.male,
                        borderRadius: 4
                    }
                ]
            };
        }
    },
    {
        type: 'doughnut',
        title: 'Répartition TV vs Radio',
        mediaType: 'all',
        getData: function(year) {
            return {
                labels: ['TV', 'Radio'],
                datasets: [{
                    data: [45.2, 43.8],
                    backgroundColor: [TV_COLORS.female, TV_COLORS.radio],
                    borderWidth: 3,
                    borderColor: '#ffffff',
                    hoverOffset: 10
                }]
            };
        }
    }
];

// Générer des données fictives
function generateMockTVData() {
    const channels = ['TF1', 'France 2', 'France 3', 'ARTE', 'M6', 'Canal+'];
    const data = channels.map(() => Math.floor(35 + Math.random() * 25));
    return {
        labels: channels,
        datasets: [
            {
                label: 'Femmes',
                data: data,
                backgroundColor: TV_COLORS.female,
                borderRadius: 4
            },
            {
                label: 'Hommes',
                data: data.map(v => 100 - v),
                backgroundColor: TV_COLORS.male,
                borderRadius: 4
            }
        ]
    };
}

// Initialiser le graphique
function initMainChart() {
    const canvas = document.getElementById('mainChart');
    if (!canvas) {
        console.error('Canvas non trouvé');
        return;
    }
    
    const ctx = canvas.getContext('2d');
    
    // Détruire le graphique existant
    if (mainChart) {
        mainChart.destroy();
    }
    
    // Récupérer l'année
    const yearFilter = document.getElementById('yearFilter');
    const year = yearFilter ? yearFilter.value : (years[0] || '2019');
    
    // Obtenir la configuration et les données
    const config = chartConfigs[currentChartIndex];
    const chartData = config.getData(year);
    
    // Options communes pour TV
    const commonOptions = {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                display: config.type !== 'doughnut',
                position: 'bottom',
                labels: {
                    color: TV_COLORS.text,
                    padding: 15,
                    font: { size: 11, family: 'Segoe UI' },
                    usePointStyle: true,
                    pointStyle: 'rectRounded'
                }
            },
            title: {
                display: true,
                text: config.title,
                color: TV_COLORS.text,
                font: { size: 14, weight: '600', family: 'Segoe UI' },
                padding: { bottom: 10 }
            },
            tooltip: {
                backgroundColor: 'rgba(255, 255, 255, 0.95)',
                titleColor: TV_COLORS.text,
                bodyColor: TV_COLORS.text,
                borderColor: TV_COLORS.accent,
                borderWidth: 1,
                padding: 12,
                cornerRadius: 6,
                displayColors: true,
                titleFont: { weight: '600' }
            }
        },
        scales: config.type === 'doughnut' ? {} : {
            x: {
                grid: { color: TV_COLORS.grid },
                ticks: { 
                    color: TV_COLORS.text,
                    font: { size: 10 }
                }
            },
            y: {
                beginAtZero: true,
                max: 100,
                grid: { color: TV_COLORS.grid },
                ticks: { 
                    color: TV_COLORS.text,
                    font: { size: 10 },
                    callback: v => v + '%'
                }
            }
        },
        animation: {
            duration: 600,
            easing: 'easeOutQuart'
        },
        interaction: {
            intersect: false,
            mode: 'index'
        }
    };
    
    // Créer le graphique
    mainChart = new Chart(ctx, {
        type: config.type,
        data: chartData,
        options: commonOptions
    });
    
    console.log('Graphique créé:', config.title);
    
    // Déclencher l'effet de transition
    const container = document.querySelector('.tv-chart-container');
    if (container) {
        container.classList.add('tv-glitch');
        setTimeout(() => container.classList.remove('tv-glitch'), 300);
    }
}

// Navigation entre graphiques
function navigateChart(direction) {
    currentChartIndex += direction;
    
    if (currentChartIndex < 0) {
        currentChartIndex = chartConfigs.length - 1;
    } else if (currentChartIndex >= chartConfigs.length) {
        currentChartIndex = 0;
    }
    
    initMainChart();
    
    // Mettre à jour les stats
    if (typeof updateMathematicalStats === 'function') {
        const year = years[currentYearIndex] || '2019';
        updateMathematicalStats(year);
    }
}

// Navigation entre années
function navigateYear(direction) {
    if (years.length === 0) return;
    
    currentYearIndex += direction;
    
    if (currentYearIndex < 0) {
        currentYearIndex = years.length - 1;
    } else if (currentYearIndex >= years.length) {
        currentYearIndex = 0;
    }
    
    const yearSelect = document.getElementById('yearFilter');
    if (yearSelect) {
        yearSelect.value = years[currentYearIndex];
    }
    
    // Mettre à jour l'affichage de l'année
    const yearElement = document.getElementById('currentYear');
    if (yearElement) {
        yearElement.textContent = years[currentYearIndex];
    }
    
    initMainChart();
    
    // Mettre à jour les stats
    if (typeof updateMathematicalStats === 'function') {
        updateMathematicalStats(years[currentYearIndex]);
    }
}

// Initialiser les événements de navigation
function initChartNavigation() {
    const leftBtn = document.getElementById('chartLeft');
    const rightBtn = document.getElementById('chartRight');
    
    if (leftBtn) {
        leftBtn.addEventListener('click', () => navigateChart(-1));
    }
    
    if (rightBtn) {
        rightBtn.addEventListener('click', () => navigateChart(1));
    }
}

// Calculs statistiques
function calculateStatistics(data) {
    if (!data || data.length === 0) return null;
    
    const mean = data.reduce((sum, val) => sum + val, 0) / data.length;
    const variance = data.reduce((sum, val) => sum + Math.pow(val - mean, 2), 0) / data.length;
    const stdDev = Math.sqrt(variance);
    
    const sorted = [...data].sort((a, b) => a - b);
    const median = sorted.length % 2 === 0
        ? (sorted[sorted.length / 2 - 1] + sorted[sorted.length / 2]) / 2
        : sorted[Math.floor(sorted.length / 2)];
    
    const parityGap = Math.abs(50 - mean);
    
    // Corrélation simulée
    const correlation = 0.5 + Math.random() * 0.4;
    
    return {
        mean: mean.toFixed(2),
        variance: variance.toFixed(2),
        stdDev: stdDev.toFixed(2),
        median: median.toFixed(2),
        parityGap: parityGap.toFixed(2),
        correlation: correlation.toFixed(2)
    };
}

function updateMathematicalStats(year) {
    // Valeurs par défaut
    const defaultStats = {
        variance: '4,635',
        stdDev: '3.17',
        median: '45.3%',
        parityGap: '22.6%',
        correlation: '0.72'
    };
    
    // Si on a des données réelles, les utiliser
    if (window.CSV_DATA?.years) {
        const tvData = window.CSV_DATA.years.tv?.[year];
        const radioData = window.CSV_DATA.years.radio?.[year];
        
        const allValues = [];
        if (tvData) {
            Object.values(tvData).forEach(v => {
                if (typeof v === 'number') allValues.push(v);
            });
        }
        if (radioData) {
            Object.values(radioData).forEach(v => {
                if (typeof v === 'number') allValues.push(v);
            });
        }
        
        if (allValues.length > 0) {
            const stats = calculateStatistics(allValues);
            if (stats) {
                document.getElementById('variance').textContent = stats.variance;
                document.getElementById('stdDev').textContent = stats.stdDev;
                document.getElementById('median').textContent = stats.median + '%';
                document.getElementById('parityGap').textContent = stats.parityGap + '%';
                document.getElementById('correlation').textContent = stats.correlation;
            }
        }
    } else {
        // Utiliser les valeurs par défaut
        document.getElementById('variance').textContent = defaultStats.variance;
        document.getElementById('stdDev').textContent = defaultStats.stdDev;
        document.getElementById('median').textContent = defaultStats.median;
        document.getElementById('parityGap').textContent = defaultStats.parityGap;
        document.getElementById('correlation').textContent = defaultStats.correlation;
    }
}

// Initialiser au chargement
document.addEventListener('DOMContentLoaded', function() {
    console.log('Initialisation des graphiques TV...');
    
    setTimeout(() => {
        initYears();
        initMainChart();
        initChartNavigation();
        
        // Initialiser les stats
        if (typeof updateMathematicalStats === 'function') {
            const year = years[currentYearIndex] || '2019';
            updateMathematicalStats(year);
        }
        
        console.log('Graphiques TV initialisés');
    }, 300);
});

// Export des fonctions
window.navigateChart = navigateChart;
window.navigateYear = navigateYear;
window.initMainChart = initMainChart;
window.updateMathematicalStats = updateMathematicalStats;
window.TV_COLORS = TV_COLORS;

